﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using System.Xml.Linq;
using UnityEngine;

//ColourGroup represents a grouped selection of ColourEntries in the Colours.xml
public class ColourGroup
{
    public string Name { get; private set; }
    public Dictionary<string, ColourEntry> Colours { get; private set; } = new Dictionary<string, ColourEntry>();

    public ColourGroup()
    {

    }

    public bool Parse(XElement groupElem)
    {
        try
        {
            Name = groupElem.Attribute("Name").Value;

            var coloursXMLList = (from ele in groupElem.Descendants("Colour")
                                  select ele).ToList();    //Get all the colour elements

            for (int i = 0; i < coloursXMLList.Count; i++)
            {
                XElement colourElem = coloursXMLList[i];    //Get this specific ColourEntry
                if (colourElem.Attribute("Key") != null)
                {
                    string thisColourKey = colourElem.Attribute("Key").Value;   //We've got a key

                    if (!Colours.ContainsKey(thisColourKey))
                    {
                        Color thisParsedColour;

                        if (ColorUtility.TryParseHtmlString(colourElem.Value, out thisParsedColour))    //This key is unique, so let's parse the colour
                        {
                            ColourEntry thisColourEntry;

                            if (colourElem.Attribute("Tags") != null)   //We've got tags, call the appropriate constructor
                            {
                                string thisColourTagAttributeValue = colourElem.Attribute("Tags").Value;
                                List<string> thisColourTags = thisColourTagAttributeValue.Split('|').ToList();
                                thisColourEntry = new ColourEntry(thisColourKey, thisColourTags, thisParsedColour);
                            }

                            else    //We don't have tags, call the appropriate constructor
                            {
                                thisColourEntry = new ColourEntry(thisColourKey, thisParsedColour);
                            }

                            Colours.Add(thisColourKey, thisColourEntry);    //Now let's add the parsed colour to the list
                        }

                        else
                        {
                            Debug.LogError(string.Format("ERROR: Group colour index {0} has an invalid colour HTML string.", i));
                        }
                    }

                    else
                    {
                        Debug.LogError(string.Format("ERROR: Group colour key {0} is a duplicate.", thisColourKey));
                    }
                }

                else
                {
                    Debug.LogError(string.Format("ERROR: Group colour index {0} has no key.", i));
                }
            }

            return true;
        }

        catch (Exception ex)
        {
            Preloader.ExceptionMessage = ex.ToString();
            return false;
        }
    }

    //Returns a random ColourEntry using Randomizer
    public ColourEntry GetRandomColourEntry()
    {
        if (Colours.Count > 0)
        {
            Randomizer.Regenerate();
            ColourEntry colourEntry = Colours.ElementAt(Randomizer.RNG.Next(0, Colours.Count)).Value;
            return colourEntry;
        }

        return null;
    }
}